{------------------------------------------------------------------------------}
{ SibVRV
{ svGE Unit
{ Created 12.03.2001 by Vereshagin Roman Vladimirovich.
{ History:
{  VR - Unit Created
{------------------------------------------------------------------------------}
unit svGE_SoundObjects_Modules;
interface
uses svFiles, svGE_SoundDriverClass, svGE_SoundChannels;

type
  { MOD file structures (32 channel MOD) }
  tmodhdr = packed record
    songname : array[0..19] of char;

    sampleinfo : array[1..31] of packed record
      name       : array[0..21] of char;
      length     : word;
      finetune   : byte;
      volume     : byte;
      loopstart  : word;
      looplength : word;
    end;

    songlength : byte;
    jumppos    : byte;
    arrange    : array[0..127] of byte;
    id         : array[0..3] of char;
  end;

  tmodhdr_16 = packed record
    songname : array[0..19] of char;

    sampleinfo : array[1..15] of packed record
      name       : array[0..21] of char;
      length     : word;
      finetune   : byte;
      volume     : byte;
      loopstart  : word;
      looplength : word;
    end;

    songlength : byte;
    jumppos    : byte;
    arrange    : array[0..127] of byte;
  end;

  { S3M file structures }
  ts3mhdr = packed record
    songname    : array[0..27] of char;
    hex1Ah      : byte;
    filetype    : byte; { = 10h }
    reserved1   : array[0..1] of byte;
    songlength  : word;
    instruments : word;
    patterns    : word;
    flags       : word;
    version     : word;
    fileformat  : word;
    id          : array[0..3] of char;
    globalvol   : byte;
    starttempo  : byte;
    startbpm    : byte;
    mastervol   : byte;
    ultaclick   : byte;
    defpan      : byte;
    reserved2   : array[0..7] of byte;
    special     : word;
    channelinfo : array[0..31] of byte;
  end;

  ts3minstr = packed record
    typ       : byte;
    filename  : array[0..11] of char;
    unused    : byte;
    segment   : word;
    size      : longint;
    loopstart : longint;
    loopend   : longint;
    volume    : byte;
    disk      : byte;
    packing   : byte;
    flags     : byte;
    c2spd     : longint;
    reserved  : array[0..3] of byte;
    GrvsRamPos: word;{Internal use for GUS}
    loopexpans: word;{Internal use for SB}
    lastpos   : longint;{Internal use for SB}
    samplename: array[0..27] of char;
    kennung   : array[0..3] of char;{SCRS}
  end;

  { XM file structures }
  txmstart = packed record { the few first bytes of a XM }
    id         : array[0..16] of char; { 'Extended module: ' }
    songname   : array[0..19] of char;
    hex1Ah     : byte;
    tracker    : array[0..19] of char;
    version    : word;                 { Current version is $0103}
  end;

  txmhdr = packed record { follows the XM start }
    headersize   : longint;
    songlength   : word; { Song length in pattern order table  }

    restart      : word; { Loop position                       }
    channels     : word; { must be multiple of 2               }
    patterns     : word; { number of patterns; max 256         }
    instruments  : word; { number of instruments; max 128      }
    flags        : word; { only bit 0 used for frequency table }
    starttempo   : word;
    startbpm     : word;
  end;

  { Internal pattern format }
  tpatt = record
    per : word;
    ins : byte;
    vol : byte;
    cmd : byte;
    arg : byte;
  end;

  pxmsmpmapping = ^txmsmpmapping;
  txmsmpmapping = array[1..128,1..96] of byte;

  pmodule = ^tmodule;
  tmodule = object(TsvSoundObject)
    starttempo,
    startbpm     : byte;
    actline      : longword;
    actpattern   : longword;
    tempo        : word;
    bpm          : word;
    tick         : word;
    arrange      : array[0..255] of byte;
    arrangespeed : array[0..255] of record
                     tempo,
                     bpm  : byte;
                   end;
    pattern      : array[0..255] of record
                     p    : pointer;
                     size : word;
                     rows : byte;
                   end;
    instruments  : word;
    patterns     : longword;
    linesize     : word;
    cs           : tchannelset;
    playchnls    : longword;
    flags        : longword;
    songlength   : longword;
    songname     : string[30];
    oscillator   : longword;
    pershift     : byte;
    portashift   : byte;
    t0volslide   : boolean;
    format       : word;
    glissando    : boolean; {!! not supported yet}
    tracker      : string[47];

    constructor init(var dest : TsvSoundDriver);
    destructor  done;virtual;
    procedure   play;virtual;
    procedure   stop;virtual;
    procedure   pause;virtual;
    function    isplaying : boolean;virtual;
    function    ispaused : boolean;virtual;

    function    setpos(pos : longword) : boolean;virtual;
    function    getpos : longword;virtual;
    function    getsize : longword;virtual;

    function    change : boolean;virtual;
    { Out of interest for you: }
    procedure   callproc;virtual;
    private
    busy         : boolean;
    lastpos      : longword;
    patdelay     : word;    { Pattern delay value }
    minslide,
    maxslide     : word;    { Porta slide min/max }
    dummypatt    : pointer;
    seeking      : boolean; { Indicates seeking   }

    function    getfrequency(period : longword) : longword;virtual;

    procedure   calcline;virtual;
    procedure   calcdata_tickn0;
    procedure   calcdata_tick0;
    procedure   calcdata_tick;

    procedure   setchannels(channels : longword);virtual;
  end;

  pmod = ^tmod;
  tmod = object(tmodule)
    header       : tmodhdr;

    constructor init(var dest : TsvSoundDriver;var f : TsvFile);
  end;

  ps3m = ^ts3m;
  ts3m = object(tmodule)
    header  : ts3mhdr;
    mapping : array[0..31] of byte;
    panlist : array[0..31] of byte;

    constructor init(var dest : TsvSoundDriver;var f : TsvFile);
  end;

  pxm = ^txm;
  txm = object(tmodule)
    header  : txmhdr;

    constructor init(var dest : TsvSoundDriver;var f : TsvFile);
    destructor  done;virtual;

    private

    startinfo   : txmstart;
    amigalimits : boolean;
    mapping     : array[1..128,0..15] of byte;
    smpmapping  : pxmsmpmapping;

    function    getfrequency(period : longword) : longword;virtual;

    procedure   calcline;virtual;
  end;

const
  fmt_unknown = $0000;
  fmt_mod     = $0001;
  fmt_s3m     = $0002;
  fmt_xm      = $0003;
  fmt_it      = $0004;

implementation
uses svGE_SoundDataConvetations;


function inttostr(l : longint) : string;
{ convert longint to string }
var
  c : string;
begin
  str(l,c);
  inttostr := c;
end;


function hexstr(l : longword;width : word) : string;
{ Get from longint value hex-string }
const
  hexs : array[0..15] of char=
  ('0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F');
var
  s : string;
  i : longint;
begin
  s := '';
  for i := 0 to width-1 do
  begin
    s := hexs[l and 15]+s;
    l := l shr 4;
  end;
  hexstr := s;
end;

function makestring(var data;size : word) : string;
type
  tchar = array[0..255] of char;

var
  i : byte;
  s : string;

begin
  s := '';
  for i := 0 to size-1 do s := s+char(pointer(longword(@data)+i)^);
  makestring := s;
end;


function filtername(name : array of char;max : word) : string;
var
  s : string;
  i : integer;

begin
  s := '';
  for i := 0 to max do
  begin
    if (name[i] = #0) then break;
    if (name[i] >= #32) then
      s := s+name[i] else s := s+#32;
  end;

  filtername := s;
end;

function amiword(w : word): word;
begin
  amiword := w shr 8+(w and $FF) shl 8;
end;

constructor tmodule.init;
begin
  inherited init(dest,0);
  actline      := 0;
  actpattern   := 0;
end;

destructor tmodule.done;
var
  i : integer;

begin
  stop;
  while busy do;
  for i := 0 to patterns-1 do if (pattern[i].p <> dummypatt) then freemem(pattern[i].p,pattern[i].size);
  if (assigned(dummypatt)) then freemem(dummypatt,pattern[patterns].size);
  for i := 0 to playchnls-1 do cs.channel[i].sound.done;
  cs.done;
  inherited done;
end;

procedure tmodule.play;
begin
  { If paused, unpause }
  if (flags and snd_pause <> 0) then pause
  else { else restart from beginning }
  begin
    stop;
    tempo      := starttempo;
    bpm        := startbpm;
    setbuffersize(round(dat.mainfreq/(bpm*2/5)));
    flags      := flags or snd_play;
  end;
end;

procedure tmodule.stop;
var
  i : integer;

begin
  flags := flags and not (snd_play or snd_pause);
  for i := 0 to playchnls-1 do cs.channel[i].sound.stop;

  actline    := 0;
  actpattern := 0;
  tick       := 0;
end;

procedure tmodule.pause;
var
  i : integer;

begin
  while busy do;
  busy := true;
  if (flags and snd_pause <> 0) then
  begin
    flags := (flags and not snd_pause) or snd_play;
    for i := 0 to playchnls-1 do cs.channel[i].sound.pause;
  end
  else if (flags and snd_play <> 0) then
  begin
    flags := (flags and not snd_play) or snd_pause;
    for i := 0 to playchnls-1 do cs.channel[i].sound.pause;
  end;
  busy := false;
end;

function tmodule.isplaying : boolean;
begin
  isplaying := flags and snd_play <> 0;
end;

function tmodule.ispaused : boolean;
begin
  ispaused := flags and snd_pause <> 0;
end;

function tmodule.setpos(pos : longword) : boolean;
var
  i          : integer;
  difference : longint;

  function getpattern(pos : longword) : longword;
  { Returns pattern of pos value }
  var
    count : longword;

  begin
    count := 0;
    while (pos > longword(pattern[arrange[count]].rows)+1) do
    begin
      dec(pos,word(pattern[arrange[count]].rows)+1);
      inc(count);
    end;
    getpattern := count+1;
  end;

  function getline(pos : longword) : longword;
  { Returns line of pos value }
  var
    count : longword;

  begin
    count := 0;
    while (pos > longword(pattern[arrange[count]].rows)+1) do
    begin
      dec(pos,word(pattern[arrange[count]].rows)+1);
      inc(count);
    end;
    getline := pos;
  end;

  function getposition(patt,line : longword) : longword;
  { Returns pos value by calculating pattern and line }
  var
    count,i : longword;

  begin
    count := 0;
    for i := 0 to patt do inc(count,word(pattern[arrange[i]].rows)+1);
    getposition := count+line;
  end;

begin
  setpos := false;
  if (flags and snd_pause <> 0) then exit;

  { Fail if pattern > max pattern }
  if (pos > getsize)or(getpattern(pos)-1 > songlength) then exit;

  { Set busy flag - otherwise machine may hang! }
  while busy do;
  busy := true;

  seeking := true; { Some functions require to know this }

  if (pos <= 1) then
  begin
    for i := 0 to playchnls-1 do cs.channel[i].sound.stop;
    pos := 0;
    actpattern := 0;
    actline    := 0;
    setpos     := true;
    busy       := false;
    seeking    := false;
    exit;
  end;


  difference := longint(pos)-longint(getposition(actpattern,actline));

  if (difference < 0) then
  begin
    { Backwards: stop all channels! }
    for i := 0 to playchnls-1 do cs.channel[i].sound.stop;
    actpattern := getpattern(pos)-1;
    actline    := 0;
    for i := 0 to getline(pos)-1 do
    begin
      calcline;
      if (actline >= getline(pos)) then break;
    end;
  end
  else { Forward }
  begin
    for i := 0 to difference-1 do
    begin
      calcline;
      if (actpattern >= getpattern(pos))and(actline >= getline(pos)) then break;
    end;
  end;

  seeking := false;

  setpos  := true;
  busy    := false;
end;

function tmodule.getpos : longword;

  function getposition(patt,line : longword) : longword;
  { Returns pos value by calculating pattern and line }
  var
    count,i : longword;

  begin
    count := 0;
    for i := 1 to patt do inc(count,word(pattern[arrange[i]].rows)+1);
    getposition := count+line;
  end;

begin
  getpos := getposition(actpattern,actline);
end;

function tmodule.getsize : longword;

  function getposition(patt,line : longword) : longword;
  { Returns pos value by calculating pattern and line }
  var
    count,i : longword;

  begin
    count := 0;
    for i := 1 to patt do inc(count,word(pattern[arrange[i]].rows)+1);
    getposition := count+line;
  end;

begin
  getsize := getposition(songlength-1,pattern[arrange[songlength-1]].rows);
end;

function tmodule.change : boolean;
var
  actpos : longword;

begin
  actpos := actpattern*64*32+actline*32+tick;
  change := (actpos <> lastpos);
  lastpos := actpos;
end;


function tmodule.getfrequency(period : longword) : longword;
begin
   getfrequency := oscillator div period;
end;

procedure tmodule.calcline;
type
  tln = array[0..31] of tpatt;

var
  ln        : ^tln;
  w         : word;
  ch,i      : integer;
  dontplay  : boolean;  { don't play current note     }
  dontinc   : boolean;  { don't increase current line }
  curr      : tpatt;
  per       : longint;
  last      : word;
  patt_break,
  patt_jump : boolean;

  change_volume,
  change_freq  : boolean;

begin
  if (actline = longword(pattern[arrange[actpattern]].rows)+1) then
  begin
    actline := 0;
    inc(actpattern);
    arrangespeed[actpattern].bpm   := bpm;
    arrangespeed[actpattern].tempo := tempo;
  end;

  { Stop playback if song end reached }
  if (actpattern >= songlength) then
  begin
    stop;
    exit;
  end;

  ln := pointer(longword(pattern[arrange[actpattern]].p)+longword(actline*linesize));

  patt_break := false; { Clear pattern not broken flag }
  patt_jump  := false; { Clear pattern jump flag       }

  dontinc    := false; { Don't increase line flag      }

  { Now take each note in the line and parse it }
  for ch := 0 to playchnls-1 do with cs.channel[ch] do
  begin
    curr := ln^[ch];
    per  := curr.per;

    { Ignore porta to note if instrument different }
    if (curr.cmd = $03)and(curr.ins <> 0)and(curr.ins <> voice) then
    begin
      curr.cmd := 0;
      curr.arg := 0;
    end;

    if (per <> 0)and(per <> mdkeyoff) then
    begin
      w := curr.ins;

      if (w = 0) then w := voice;
      if (w = 0) then continue;

      if (cs.instrument[w].tuning <> 8363)and(cs.instrument[w].tuning <> 0) then
          per := 8363 * per div cs.instrument[w].tuning;

      if (curr.cmd <> $03)and(per <> 0) then
      begin
        period := per;
        freq   := {oscillator div per;} getfrequency(per)
      end;
    end;

    change_volume := false;

    { Don't play note if note delay, porta to note or key off }
    case format of
      fmt_mod : dontplay := ((curr.cmd = $0E)and((curr.arg shr 4) = $0D))or
                            ((curr.cmd = $03)and(curr.ins = 0))
                            ;
      fmt_s3m : dontplay := ((curr.cmd = $0E)and((curr.arg shr 4) = $0D))or
                            (curr.cmd = $03)or
                            (curr.per = mdkeyoff);
    end;

    { Strange thing... reset volume if instrument, but not period given }
    if (curr.ins <> 0)and(curr.per = 0)and(voice <> 0) then
    begin
      volume := cs.instrument[voice].volume;
      change_volume := true;
    end

    else

    if (not dontplay) then
    if (curr.ins <> 0) then
      cs.play(ch,curr.ins,freq,cs.instrument[curr.ins].volume) else
              if (per <> 0)or(curr.ins <> 0)and(per = 0) then
              cs.play(ch,voice,freq,volume);

    { Retrigger volume if porta note and same instrument on line }
    if (curr.cmd = $03)and(curr.ins <> 0)and(curr.ins = voice)and
       (format <> fmt_mod)and(curr.vol = 0)
       then curr.vol := cs.instrument[voice].volume+$10;

    if ((arpeggio)or(vibrato))and(period <> 0) then
    begin
      cs.setfrequency(ch,freq);
      arpeggio    := false;
      vibrato     := false;
    end;

    { Reset all FX which are linewise }
    tremolo      := false;
    porta        := false;
    bfinepslide  := false;
    bfinevslide  := false;
    bvolumeslide := false;
    bportaslide  := false;
    retrig       := 0;
    tremor       := false;
    last         := lasteffect;
    lasteffect   := curr.cmd+curr.arg shl 8;

    if (curr.vol <> 0)and(curr.vol <= $50) then
    begin
      volume := curr.vol-$10;
      if ((curr.cmd = $0E)and((curr.arg shr 4) = $0D))
      then else change_volume := true;
    end;

    { Key off means to set volume to 0 }
    if (curr.per = mdkeyoff) then
    begin
      volume := 0;
      change_volume := true;
    end;

    case curr.cmd of
      $00 : if (curr.arg <> 0) then
            begin{ok; Arpeggio          }
              arpeggio  := true;
              arpeggiox := (curr.arg shr 4);
              arpeggioy := (curr.arg and $F);
            end else lasteffect := mdkeyoff;
      $01 : begin{ok; Portamento up     }
              portaslide  := -(integer(curr.arg) shl portashift);
              bportaslide := true;
            end;
      $02 : begin{ok; Portamento down   }
              portaslide  := integer(curr.arg) shl portashift;
              bportaslide := true;
            end;
      $03 : begin{ok; Porta to note     }
              if (per <> 0) then portanote := per;
              if (curr.arg <> 0) then
              begin
                portaspeed := word(curr.arg) shl portashift;
                if (period > portanote) then portaspeed := -portaspeed;
              end;
              porta := (portanote <> 0);
            end;
      $04 : begin{ok; Set vibrato       }
              if (curr.arg and $F0 <> 0) then vibratospeed := curr.arg shr 4;
              if (curr.arg and $F <> 0) then vibratodepth := (curr.arg and $F) shl pershift;
              vibrato := true;
            end;
      $05 : begin{ok; Porta+vol slide   }
              if (curr.arg and $F <> 0) then volumeslide := -(curr.arg and $F)
                else if (curr.arg and $F0 <> 0) then volumeslide := curr.arg shr 4;
              porta := (portanote <> 0);
              bvolumeslide := true;
            end;
      $06 : begin{ok; Vibrato+vol slide }
              if (curr.arg and $F <> 0) then volumeslide := -(curr.arg and $F)
              else if (curr.arg and $F0 <> 0) then volumeslide := curr.arg shr 4;
              vibrato := true;
              bvolumeslide  := true;
            end;
      $07 : begin{ok; Tremelo           }
              if (curr.arg and $F0 <> 0) then tremolospeed := curr.arg shr 4;
              if (curr.arg and $F <> 0) then tremolodepth := curr.arg and $F;
              tremolo := true;
            end;
      $08 : begin{ok; Panning           }
              panning := integer(curr.arg)*2-256;
              change_volume := true;
              {!! note: fmoddoc says values are from 00..$80, $A4 for Dolby background
                modfil11 instead talks of 0..255 ... the only mod I have seemed to
                use $80 for mod... other checks should be performed
              }
            end;
      $09 : begin{ Set instrument offset }
              if (curr.arg <> 0) then lastsetpos := longint(curr.arg) shl 8;
              cs.setpos(ch,lastsetpos);
            end;
      $0A : begin { Volume slide }
              if (curr.arg and $F <> 0) then volumeslide := -(curr.arg and $F)
              else volumeslide := curr.arg shr 4;
              bvolumeslide  := true;
            end;
      $0B : begin { Position jump }
              if (curr.arg > songlength-1) then curr.arg := 0; { Means 'restart' - could stop here }
              actline    := 0;
              actpattern := curr.arg;
              dontinc    := true;
              patt_jump  := true;
            end;
      $0C : begin{ok; Set channel volume }
              if (curr.arg > 64) then curr.arg := 64;
              volume := curr.arg;
              change_volume := true;
            end;
      $0D : begin{ok; Pattern break      }
              if (not patt_jump)and(not patt_break) then
                if (actpattern < songlength-1) then inc(actpattern) else actpattern := 0;

              actline := 10*(curr.arg shr 4)+curr.arg and $0F;
              if (actline > 63) then actline := 0;
              dontinc    := true;
              patt_break := true;
            end;
      $0E : case (curr.arg shr 4) of
              $01 : begin { ok; Fine portamento up   }
                      finepslide := -((curr.arg and $F) shl portashift);
                      bfinepslide := (finepslide <> 0);
                    end;
              $02 : begin { ok; Fine portamento down }
                      finepslide := (curr.arg and $F) shl portashift;
                      bfinepslide := (finepslide <> 0);
                    end;
              $03 : glissando := curr.arg and 1 = 1; { glissando control }
              $04 : begin { Set vibrato waveform }
                      vibratotype := curr.arg and 7;
                      if (vibratotype and 3) = 3 then
                        vibratotype := random(2)+curr.arg and 4;
                    end;
              $05 : begin { Set finetune }
                      cs.instrument[voice].tuning := getc2spd(curr.arg and $F);
                    end;
              $06 : if (not seeking) then
                    if (not patternloop.active) then  { pattern loop }
                    begin
                       if (curr.arg and $F = 0) then { We have to set the loop start }
                       begin
                         patternloop.line     := actline;
                         patternloop.pattern  := actpattern;
                       end else { we have the loop end }
                       begin
                         patternloop.active   := true;     { This causes that a new set of loop start overwrites old one }
                         patternloop.count    := curr.arg and $F;
                         actline              := patternloop.line;
                         dontinc              := true;
                       end;

                    end else
                    if (curr.arg and $F <> 0) then { We already are repeating, do jump/count down }
                    begin
                       dec(patternloop.count);
                       if (patternloop.count > 0) then
                       begin
                         actline := patternloop.line;
                         dontinc := true;
                       end else patternloop.active := false;
                    end;

              $07 : begin { Set tremolo waveform }
                      tremolotype := curr.arg and 7;
                      if (tremolotype and 3) = 3 then
                        tremolotype := random(2)+curr.arg and 4;
                    end;
              $08 : begin { ok; Position panning }
                      panning := (curr.arg and $F);
                      {if (panning > 7) then inc(panning);}
                      panning := panning*34-256;
                      change_volume := true;
                    end;
              $09 : begin { ok; Retrigger note   }
                      retrig     := (curr.arg and $F);
                      retrigtype := 0;
                    end;
              $0A : begin { ok; Fine volume slide up   }
                      finevslide := curr.arg and $F;
                      bfinevslide := (finevslide <> 0);
                    end;
              $0B : begin { ok; Fine volume slide down }
                      finevslide := -integer(curr.arg and $F);
                      bfinevslide := (finevslide <> 0);
                    end;
              $0C : begin { ok; Cut note }
                      if (curr.arg and $F <> 0) then volcutcount := (curr.arg and $F);
                    end;
              $0D : begin { ok; Note delay }
                      notedelay := (curr.arg and $F);
                      if (curr.ins <> 0) then voice := curr.ins;
                      if (curr.vol = 0) then volume := cs.instrument[voice].volume;
                    end;
              $0E : begin { ok; Pattern delay }
                      patdelay := (curr.arg and $F);
                    end;
              $0F :; { Invert loop; generally not supported on PC }
            end;
      $0F : if (curr.arg <> 0) then
            begin{ ok; Set bpm/tempo }
              if (curr.arg < 32) then tempo := curr.arg
              else
              begin
                bpm := curr.arg;
                setbuffersize(round(dat.mainfreq/(bpm*2/5)));
              end;
            end;

      { additional converted S3M stuff }
      $10 : begin { Set global volume }
              if (curr.arg > 64) then curr.arg := 64;
              cs.globalvol := curr.arg*4;
              for i := 0 to playchnls-1 do cs.setvolume(i,cs.channel[i].volume);
            end;
      $1B : begin
              retrig     := (curr.arg and $F);
              retrigtype := (curr.arg shr 4);
            end;
      $1D : begin { Tremor }
              tremor      := true;
              tremorcount := 0;
              if (curr.arg <> 0) then
              begin
                tremorx     := curr.arg shr 4+1;
                tremory     := tremorx+curr.arg and 7+1;
              end; { else tremorx and y should hold our last values }
            end;
      $21 : case (curr.arg and $80) of
              $00 :  begin { ok; Extra fine portamento up   }
                      if (curr.arg and $7F <> 0) then
                      finepslide := -(curr.arg and $7F);
                      bfinepslide := (finepslide <> 0);
                    end;
              $80 : begin { ok; Extra Fine portamento down }
                      if (curr.arg and $7F <> 0) then
                      finepslide := curr.arg and $7F;
                      bfinepslide := (finepslide <> 0);
                    end;
            end;

      $22 : begin { S3M Volume slide }
              if (curr.arg <> 0) then lvolumeslide := curr.arg;

              if (lvolumeslide and $F = $F)and(lvolumeslide shr 4 <> 0) then
              begin { Fine v slide up }
                finevslide  := (lvolumeslide shr 4) and $F;
                bfinevslide := (finevslide <> 0);
              end
              else
              if (lvolumeslide and $F0 = $F0)and(lvolumeslide and $F <> 0) then
              begin { Fine v slide down }
                finevslide := -(lvolumeslide and $F);
                bfinevslide := (finevslide <> 0);
              end
              else { Normal volume slide}
              begin
                if (curr.arg <> 0) then
                if (lvolumeslide and $F <> 0) then volumeslide := -(lvolumeslide and $F)
                else if (lvolumeslide and $F0 <> 0) then volumeslide := lvolumeslide shr 4;
                bvolumeslide := volumeslide <> 0;
              end;
            end;

      $23 : begin { S3M Portamento down   }
              if (curr.arg = 0) then curr.arg := lportaslide else lportaslide := curr.arg;

              case curr.arg shr 4 of
                $F: begin
                      if (curr.arg and $F <> 0) then
                      finepslide := (curr.arg and $F)*4;
                      bfinepslide := (finepslide <> 0);
                    end;
                $E: begin
                      if (curr.arg and $F <> 0) then
                      finepslide := (curr.arg and $F);
                      bfinepslide := (finepslide <> 0);
                    end;
                else
                    begin
                      if (curr.arg <> 0) then
                      portaslide  := (integer(curr.arg) shl portashift);
                      bportaslide := portaslide <> 0;
                    end;
              end;
            end;

      $24 : begin { S3M Portamento up     }
              if (curr.arg = 0) then curr.arg := lportaslide else lportaslide := curr.arg;

              case curr.arg shr 4 of
                $F: begin
                      if (curr.arg and $F <> 0) then
                      finepslide := -(curr.arg and $F)*4;
                      bfinepslide := (finepslide <> 0);
                    end;
                $E: begin
                      if (curr.arg and $F <> 0) then
                      finepslide := -(curr.arg and $F);
                      bfinepslide := (finepslide <> 0);
                    end;
                else
                    begin
                      if (curr.arg <> 0) then
                      portaslide  := -(integer(curr.arg) shl portashift);
                      bportaslide := portaslide <> 0;
                    end;
              end;
            end;

      $F4 : begin{ S3M Fine vibrato    }
              if (curr.arg and $F0 <> 0) then vibratospeed := curr.arg shr 4;
              if (curr.arg and $F <> 0) then vibratodepth := curr.arg and $F;
              vibrato := true;
            end;
      $F5 : if (curr.arg <> 0) then tempo := curr.arg; { Set S3M tempo }
      else lasteffect := mdnotavail;
    end;
    if (change_volume) then cs.setvolume(ch,volume);

  end;
  if (not dontinc) then
  begin
    inc(actline);
    { Reset pattern loop data if required }
    if (actline = 0)or(seeking) then for ch := 0 to playchnls-1 do with cs.channel[ch] do
      patternloop.active := false;
  end;
end;


procedure tmodule.calcdata_tick0;
var
  i  : integer;

begin
  while busy do;
  busy := true;

  for i := 0 to playchnls-1 do with cs.channel[i] do
  if (voice <> 0) then
  begin

    if (bfinevslide) then
    begin
      inc(volume,finevslide);
      if (volume < 0) then
      begin
        bfinevslide := false;
        volume      := 0;
      end else
      if (volume >= 64) then
      begin
        bfinevslide := false;
        volume      := 64;
      end;
      cs.setvolume(i,volume);
    end;

    if (bfinepslide)and(period <> 0) then
    begin
      inc(period,finepslide);
      if (period >= maxslide) then
      begin
        period      := maxslide;
        bfinepslide := false;
      end else
      if (period <= minslide) then
      begin
        period      := minslide;
        bfinepslide := false;
      end;
      freq := {oscillator div period;} getfrequency(period);
      cs.setfrequency(i,freq);
    end;
  end;
  busy := false;
end;

procedure tmodule.calcdata_tick;
Procedure GetNextX (A: Byte; CH: Integer; P1, P2: Byte);
Begin
  With CS.Channel [CH].EnvelopeCH [A],
       CS.Channel [CH].EnvelopeCH [A].Envelope DO
  Begin
    if (p2 < 12)and(p1 < 12) then
    begin
      ECounter := longint(EData [P2].X) - longint(EData [P1].X);
      DestX    := ECounter;
      DestY    := longint(EData [P1].Y) - longint(EData [P2].Y);
    end else {!! We shouldn't get here... check docu }
    begin
      ecounter := 1;
      DestX    := 1;
      DestY    := 1;
    end;
  End; { With }
End; { Procedure }
var

  Count:  Byte;
  i,vol : integer;

begin
  for i := 0 to playchnls-1 do with cs.channel[i] do
  begin

    if (format >= fmt_xm) then
    begin
      { << Added by Raid >> }

      For Count := 0 TO 1 DO
      With CS.Channel[I].EnvelopeCH[Count],
           CS.Channel[I].EnvelopeCH[Count].Envelope DO
      Begin

        IF (EStat And 1 <> 0) And (Active) Then
        Begin
          IF ECounter > 0 Then Dec(ECounter)
          Else
          IF (EStat And 2 = 0) Then
          Begin
            { // No Sustain, since Sustain is first // }
            IF (EStat And 4 <> 0) Then
            Begin
              { // Loop // }
              GetNextX(Count, I, CurPoint, CurPoint + 1);
              Inc(CurPoint);
              IF CurPoint > LEnd Then
              Begin
                CurPoint := LStart;
                GetNextX(Count, I, CurPoint, CurPoint + 1);
                Inc(CurPoint); {!! I'm not sure if this is right... }
              End; { IF }
            End { IF }
            Else
            Begin
              { // No Sustain // }
              GetNextX(Count, I, CurPoint, CurPoint + 1);
              IF CurPoint+1 < Points Then Inc(CurPoint) Else Active := False;
            End; { Else }
          End { IF }
          Else IF EStat And 2 <> 0 Then
          Begin
            { // Sustain // }
            CurY := EData[CurPoint].Y;
            IF (CurPoint < Sustain) OR (KeyFading) Then
            Begin
              GetNextX(Count, I, CurPoint, CurPoint + 1);
              IF CurPoint+1 < Points Then Inc(CurPoint) Else Active := False;
            End; { IF }
          End; { Else }


          IF Not Active Then
          Begin
            DestX     := 0;
            CurY      := EData[CurPoint].Y;
          End; { IF }

          { // Calculate Line // }

          IF (DestX > 0) Then
            CurY := EData[CurPoint].Y +
                    (ECounter * DestY) div destx;

          Case Count OF
            0: Begin
              EnvVol := CurY*4;
              CS.SetVolume(I, Volume);
            End; { 0 }
            1: Begin
               EnvPan := (CurY - 32) * 8;
               CS.SetVolume (I, Volume );
            End; { 1 }
          End; { Case }

        End { IF }
        else if (EStat and 1 = 0) then
        begin
          {envvol := 256;
          CS.SetVolume(I, Volume);}
        end;

      End; { With }

      { << End >> }

      IF (KeyFading) Then
      Begin
        Dec(FadeOutVol, VolFade);
        IF (FadeOutVol <= 0) Then
        Begin
          volume     := 0;
          fadeoutvol := 65536;
          keyfading  := false;
        End; { IF }
        CS.SetVolume(I,Volume);
      End; { IF }

    end;

    if (retrig <> 0)and(tick mod retrig = 0) then
    begin
      vol := cs.channel[i].volume;
      case retrigtype of
        $0,$8 :;
        $1    : dec(vol,1);
        $2    : dec(vol,2);
        $3    : dec(vol,4);
        $4    : dec(vol,8);
        $5    : dec(vol,16);
        $6    : vol := vol shl 1 div 3;
        $7    : vol := vol shr 1;
        $9    : inc(vol,1);
        $A    : inc(vol,2);
        $B    : inc(vol,4);
        $C    : inc(vol,8);
        $D    : inc(vol,16);
        $E    : vol := vol*3 div 2;
        $F    : vol := vol shl 1;
      end;
      if (vol < 0) then vol := 0
      else if (vol > 64) then vol := 64;
      cs.channel[i].volume := vol;
      if (voice <> 0) then cs.play(i,voice,freq,cs.channel[i].volume);
    end;

    if (format >= fmt_xm) then
    begin
      { << Added by Raid >> }

      IF BPanSlide Then
      Begin

        IF PanCounter > 0 Then Dec (PanCounter) Else BPanSlide := False;

        IF (PanSlide > 0) Then
        Begin
          IF CS.Channel [I].Panning < 256 Then Inc (CS.Channel [I].Panning, 16)
            Else BPanSlide := False;
        End { IF }
        Else
        Begin
          IF CS.Channel [I].Panning > -256 Then Dec (CS.Channel [I].Panning, 16)
            Else BPanSlide := False;
        End; { Else }

        CS.SetVolume (I, CS.Channel [I].Volume);

      End; { IF }

      IF (CS.BGVolSlide) Then
      Begin
        IF CS.GVolCounter > 0 Then Dec (CS.GVolCounter)
          Else CS.BGVolSlide := False;
        IF CS.GlobalVol < 256 Then Inc (CS.GlobalVol, CS.GVolSlide Shl 2);
        CS.SetVolume (I, CS.Channel [I].Volume);
      End; { IF }

      { << End >> }
    end;


    if (bvolumeslide)and((tick <> 0)or(t0volslide)) then
    begin
      inc(volume,volumeslide);
      if (volume <= 0) then
      begin
        bvolumeslide := false;
        volume := 0;
      end else
      if (volume >= 64) then
      begin
        volumeslide := 0;
        volume      := 64;
      end;
      cs.setvolume(i,volume);
    end;

    if (tremorx>0) and (tremory>0) then
    if (tremor)and((tremorcount mod tremorx = 0)or(tremorcount mod tremory = 0)) then
    begin
      if (tremorcount mod tremorx = 0) then
        cs.setvolume(i,cs.channel[i].volume)
        else cs.setvolume(i,0);
    end;
    inc(tremorcount);
  end;
end;

procedure tmodule.calcdata_tickn0;
var
  i               : integer;
  delta,delta_vib : longint;
  change_freq     : boolean;

begin
  while busy do;
  busy := true;

  for i := 0 to playchnls-1 do with cs.channel[i] do
  begin
    delta_vib   := 0;
    change_freq := false;

    if (notedelay <> 0)and(tick = notedelay) then
    begin
      notedelay := 0;
      if (voice <> 0) then cs.play(i,voice,freq,volume);
    end;

    if (period <> 0) then
    begin
      if (bportaslide) then
      begin
        inc(period,portaslide);
        if (period >= maxslide) then
        begin
          period      := maxslide;
          bportaslide := false;
        end else
        if (period <= minslide) then
        begin
          period      := minslide;
          bportaslide := false;
        end;
        freq := {oscillator div period;}getfrequency(period);
        change_freq := true;
      end;

      if (porta) then
      begin
        Inc(Period,PortaSpeed);
        IF (PortaSpeed < 0)And(Period < PortaNote) Then
        Begin
          Porta  := False;
          Period := PortaNote;
        End { IF }
        Else IF (PortaSpeed > 0)And(Period > PortaNote) Then
        Begin
          Porta  := False;
          Period := PortaNote;
        End; { ElseIF }
        freq := {oscillator div period;}getfrequency(period);
        change_freq := true;
      end;

      if (vibrato) then
      begin
        case vibratotype of
          0 : delta := vibrato_sine[vibratopos];
          1 : begin
                delta := (vibratopos and 31) shl 3;
                if (vibratopos > 31) then delta := 255-delta;
              end;
          2 : if (vibratopos < 32) then delta := 255 else delta := -255;
        end;

        { << Changed by Raid >> }
        { Was delta := (delta * vibratodepth) div 128; }
        Delta := (Delta * VibratoDepth) div 256;
        { << End >> }
        inc(delta_vib,delta);
        vibratopos := (vibratopos + vibratospeed) and 63;
      end;

      if (arpeggio) then
      begin
        case tick mod 3 of
          0 : cs.setfrequency(i,freq);
          1 : cs.setfrequency(i,getfrequency(getnewhalftone(period,arpeggiox)));
          2 : cs.setfrequency(i,getfrequency(getnewhalftone(period,arpeggioy)));
        end;
      end;

    end;

    { XM auto vibrato }
    if (avidepth <> 0) then
    begin
      { Vibrato waveform }
      case avitype of
        0 : begin
              { << Changed by Raid >> }
              { Was delta := xm_sine[avipos and $7F]; }
              Delta := XM_Sine[AviPos and $7F] shr 2;
              { << End >> }
              if (avipos <= 127) then delta := -delta;
            end;
        1 : if (vibratopos < 127) then delta := -255 else delta := 255;
        2 : delta := 256-(avipos shl 1);
        3 : delta := (avipos shl 1)-256;
      end;

      delta := (delta * avidepth) div 256;

      { Vibrato sweep    }
      if (avispos < avisweep) then
      begin
        delta := delta*avispos div avisweep;
        inc(avispos);
      end;

      inc(delta_vib,delta);
      avipos := (avipos+avispeed) and $FF;
    end;

    if (tremolo) then
    begin
      case tremolotype and 3 of
        0 : delta := (vibrato_sine[tremolopos]*tremolodepth);
        1 : begin
              delta := (tremolopos and 31) shl 3;
              if (tremolopos > 31) then delta := 255-delta;
            end;
        2 : if (tremolopos < 32) then delta := 255 else delta := -255
      end;

      {!! Does this code make sense?! }
      delta := volume+(delta div 64);
      if (volume < 0) then volume := 0;
      if (volume > 64) then volume := 64;
      cs.setvolume(i,volume+delta);
      tremolopos := (tremolopos+tremolospeed) and 63;
    end;


    if (volcutcount > 0) then
    begin
      dec(volcutcount);
      if (volcutcount = 0) then
      begin
        cs.channel[i].volume := 0;
        cs.setvolume(i,0);
      end;
    end;

    if (change_freq)or(delta_vib <> 0) then cs.setfrequency(i,getfrequency(period + delta_vib));

  end;
  busy := false;
end;

procedure tmodule.setchannels(channels : longword);
{ Sets number of channels; most channels in MODs don't play at the same time
  at full volume. On basis of this, we have to calculate the number of
  'divisions' (nothing else is another channel) differently  }
begin
  if (playchnls < 4) then TsvSoundObject.setchannels(playchnls)
  else TsvSoundObject.setchannels(round(playchnls/sqrt(playchnls*3)));
end;


procedure tmodule.callproc;
begin
  if (flags and snd_play <> 0)and(not busy) then
  begin
    tick := (tick+1) mod tempo;
    if (tick = 0) then
    begin
      if (patdelay > 0) then dec(patdelay) else calcline;
      calcdata_tick0;
    end else
      calcdata_tickn0;
      calcdata_tick;
  end;
end;

constructor tmod.init;
type
  tln = array[0..31] of tpatt;
const
  ascii : set of char=['a'..'z','A'..'Z',#32..#64,#91..#96,#123..#126];

var
  ln       : ^tln;

  s,s2     : string;
  i        : longword;
  j        : integer;
  line     : longword;
  w        : word;
  p        : pointer;
  l        : longword;
  pt       : pointer;
  hd16     : tmodhdr_16;
  valid    : boolean;

begin
  inherited init(dest);

  f.svFileIO(header,sizeof(header));
  s := makestring(header.id,4);

  instruments := 31;
  { Check if file is 8 channel .WOW (Grave Composer) }
  if (s = 'M.K.') then
  begin
    { Check number of patterns }
    patterns := 0;
    for i := 0 to 127 do if (header.arrange[i] > patterns) then
      patterns := header.arrange[i];
    inc(patterns);
    { Check overall sample size}
    l := 0;
    for i := 1 to 31 do
      l := l+longword(amiword(header.sampleinfo[i].length))*2;

    l := l+sizeof(header)+longword(patterns)*8*4*$40;
    if (abs(longint(f.svFileSize)-l) < 256) then s := 'WOW!';
  end;
  { Try to detect module format and set channels }
  if (s = 'TDZ1') then playchnls := 1 else
  if (s = 'TDZ2') then playchnls := 2 else
  if (s = 'TDZ3') then playchnls := 3 else
  if (s = 'M.K.')or(s = 'M!K!')or(s = 'FLT4')or(s = 'N.T.')or(s = 'EX04') then
    playchnls := 4 else
  if (s = 'OCTA')or(s = 'CD81'){or(s = 'FLT8')}or(s = 'EX08')or(s = 'WOW!') then { no FLT8 support yet... have no files }
    playchnls := 8 else
  if (pos('CHN',s) <> 0) then
  begin
    s2 := s;
    delete(s2,2,3);
    val(s2,i,j);
    if (j <> 0) then
    begin
      TsvSoundObject.done;
      fail;
    end;
    playchnls := i;
  end else
  if (pos('CH',s) <> 0) then
  begin
    s2 := s;
    delete(s2,3,2);
    val(s2,i,j);
    if (j <> 0) then
    begin
      TsvSoundObject.done;
      fail;
    end;
    playchnls := i;
  end
  else { Try to detect if it's a 15 instrument file }
  begin
    f.position := 0;
    f.svFileIO(hd16,sizeof(hd16));
    { No trivia: we have no ID... so check if parameters
      in allowed range... yes: we _possibly_ have a 15 inst mod  }
    valid := true;
    i := 0;
    while (i <= 19)and(hd16.songname[i] <> #0) do
    begin
      valid := valid and (hd16.songname[i] in ascii);
      inc(i);
    end;
    for i := 0 to 127 do if (hd16.arrange[i] > 63) then valid := false;
    valid := valid and (hd16.jumppos <= 127);

    if (valid) then { Do further tests ... }
    begin
      { Check number of patterns }
      patterns := 0;
      for i := 0 to 127 do if (hd16.arrange[i] > patterns) then
        patterns := hd16.arrange[i];
      inc(patterns);
      { Check overall sample size}
      l := 0;
      for i := 1 to 15 do l := l+longword(amiword(hd16.sampleinfo[i].length))*2;

      l := l+sizeof(hd16)+longword(patterns)*4*4*$40;
      valid := (abs(longint(f.svFileSize)-longint(l)) < 2048);
    end;

    if (valid) then
    begin
      fillchar(header,sizeof(header),0);
      move(hd16.songname,header.songname,sizeof(hd16.songname));
      move(hd16.sampleinfo,header.sampleinfo,sizeof(hd16.sampleinfo)); { some may hate me for this ... }
      move(hd16.arrange,header.arrange,sizeof(hd16.arrange));
      header.songlength := hd16.songlength;
      header.jumppos    := hd16.jumppos;
      playchnls         := 4;
      instruments       := 15;
      s := '15IN'; { Just internal }
    end
    else  { no valid mod }
    begin
      TsvSoundObject.done;
      fail;
    end;
  end;

  if (s = 'M.K.') then tracker := 'Protracker' else
  if (s = 'M!K!') then tracker := 'Protracker 2.3+' else
  if (s = 'WOW!') then tracker := 'Grave Composer' else
  if (s = 'N.T.') then tracker := 'Noisetracker' else
  if (s = 'CD81') then tracker := 'Falcon' else
  if (pos('FLT',s) <> 0) then tracker := 'Startrekker' else
  if (pos('CH',s) <> 0)and(playchnls and 1 <> 0) then tracker := 'TakeTracker' else
  if (pos('CH',s) <> 0) then tracker := 'FastTracker' else
{  if (s = '15IN') then tracker := 'Noisetracker (old)'}
  if (s = '15IN') then tracker := 'Soundtracker'
  else tracker := 'Unknown';

  { a 16 instrument file is probably written by "Noise Tracker"}

  setchannels(playchnls);
  setbuffersize(dat.mainfreq div 50);

  { Set format restrictions }
  format      := fmt_mod;
  oscillator  := osc_freq;
  pershift    := 0;
  portashift  := 0;
  minslide    := 113;
  maxslide    := 856;
  linesize    := playchnls*sizeof(tpatt);
  starttempo  := 6;
  startbpm    := 125;

  arrangespeed[0].bpm   := starttempo;
  arrangespeed[0].tempo := startbpm;

  songname := filtername(header.songname,19);

  cs.init;
  for i := 0 to maxchnls-1 do cs.channel[i].panning := channelval[i]*512-256;
  for i := 0 to maxchnls-1 do if (cs.channel[i].panning < 0) then
    inc(cs.channel[i].panning,stereomix) else dec(cs.channel[i].panning,stereomix);

  for i := 0 to playchnls-1 do cs.channel[i].sound.init(self);

  patterns := 0;
  for i := 0 to 127 do if (header.arrange[i] > patterns) then
    patterns := header.arrange[i];
  inc(patterns);

  songlength := header.songlength;
  move(header.arrange,arrange,128);
  { Load and convert all patterns }
  getmem(pt,playchnls*4*64);

  for i := 0 to patterns-1 do
  begin
    with pattern[i] do
    begin
      size := linesize*64;
      rows := 63;
      getmem(p,size);
    end;

    f.svFileIO(pt^,playchnls*4*64);

    for line := 0 to 63 do
    begin
      ln := pointer(longword(pattern[i].p)+line*linesize);
      for j := 0 to playchnls-1 do with ln^[j] do
      begin
        l   := longword(pointer(longword(pt)+line*playchnls*4+longword(j)*4)^);
        ins := (l and $10)+(l shr 20) and $F;
        per := amiword(l and $FFFF) and $0FFF;
        cmd := (l shr 16) and $F;
        arg := (l shr 24);
        vol := 0;
      end;
    end;
  end;

  freemem(pt,playchnls*4*64);

  { Load instrument settings and names }
  for i := 1 to instruments do with cs.instrument[i] do
  begin
    name := '';
    for j := 0 to 21 do
    if (header.sampleinfo[i].name[j] >= #32) then
      name := name+header.sampleinfo[i].name[j] else name := name+#32;

    size      := longint(amiword(header.sampleinfo[i].length)) shl 1;
    loopstart := longint(amiword(header.sampleinfo[i].loopstart)) shl 1;
    l := longint(amiword(header.sampleinfo[i].looplength)) shl 1;
    if (l = 2) then l := 0;
    loopend   := loopstart+l;
    if (loopend = loopstart) then
    begin
      loopend   := 0;
      loopstart := 0;
    end;
    volfade := 65536;
    volume := header.sampleinfo[i].volume;
    tuning  := getc2spd(header.sampleinfo[i].finetune and $F);
    if (loopend > size)or(loopend = 2) then loopend := 0;
    if (size = 2) then size := 0;
    ofreq := stdfreq;
  end;

  { Load and convert instrument samples }
  for i := 1 to instruments do with cs.instrument[i] do
  if (size > 0) then
  with cs.instrument[i] do
  begin
    getmem(s,size shl 1);
    getmem(p,size);
    f.svFileIO(p^,size);
    open      := true;
    aud_s8_to_s16(p,s,size);
    freemem(p);
  end;
  initialized := true;
end;

constructor ts3m.init;
type
  tln = array[0..31] of tpatt;

  ts3mpatt = record
    o_n : byte;
    ins : byte;
    vol : byte;
    cmd : byte;
    arg : byte;
  end;
  tba = array[0..32*6*64] of byte;

var
  ln             : ^tln;
  s3minstpointer : array[0..99] of word;
  s3mpattpointer : array[0..255] of word;
  i              : integer;
  s3minstrument  : ts3minstr;
  p              : pointer;
  patt           : tpatt;
  line,w,pos,ch  : word;
  data           : ^tba;
  s              : string;
  b,x,y          : byte;
  channeluse     : array[0..31] of boolean;

begin
  inherited init(dest);

  { Check for SCRM id  }
  f.svFileIO(header,sizeof(header));
  if (makestring(header.id,4) <> 'SCRM') then
  begin
    TsvSoundObject.done;
    fail;
  end;

  { Set song variables }
  songlength  := header.songlength;  { Inaccurate; fixed later }
  instruments := header.instruments;
  patterns    := header.patterns;

  { Load arrangement and pattern pointers }
  f.svFileIO(arrange,songlength);
  f.svFileIO(s3minstpointer,(header.instruments)*2);
  f.svFileIO(s3mpattpointer,header.patterns*2);

  { Check for Pan table and load if available }
  if (header.defpan = $FC) then f.svFileIO(panlist,32)
  else
    for i := 0 to 31 do panlist[i] := 0;

  t0volslide := (header.flags and $40 <> 0)or(header.version = $1300);

  songname := filtername(header.songname,27);

  { Try to detect Tracker name }
  case header.version of
    $1300 : tracker := 'Scream Tracker 3.00';
    $1301 : tracker := 'Scream Tracker 3.01';
    $1303 : tracker := 'Scream Tracker 3.03';
    $1320 : tracker := 'Scream Tracker 3.20';
    $2100 : tracker := 'Imago Orpheus 1.0';
    else  tracker := 'Unknown';
  end;

  { Set format restrictions }
  format     := fmt_s3m;
  startbpm   := header.startbpm;
  starttempo := header.starttempo;
  if (startbpm = 0) then startbpm := 125;
  if (starttempo = 0) then starttempo := 1;
  oscillator := s3m_freq;
  pershift   := 2;
  portashift := 2;

  arrangespeed[0].tempo := starttempo;
  arrangespeed[0].bpm   := startbpm;

  if (header.flags and $10 <> 0) then
  begin { Force Amiga limits }
    minslide   := 113*4;
    maxslide   := 856*4;
  end else
  begin
    minslide   := 56;
    maxslide   := 27392;
  end;
  { Filter out unused patterns }
  for i := 0 to songlength-1 do if (arrange[i] >= patterns)and(i < 255) then
  begin
    move(arrange[i+1],arrange[i],255-i);
    dec(songlength);
  end;

  for i := 0 to songlength-1 do if (arrange[i] >= patterns)or(arrange[i] >= 254) then break;
  songlength := i;
  { Check for any illegaly assigned patterns }
  getmem(data,32*6*64+6);
  { Filter out unused channels }
  for i := 0 to patterns-1 do if (s3mpattpointer[i] <> 0) then
  begin
    f.Position:=(longint(s3mpattpointer[i])*16{,soFromBeginning});
    f.svFileIO(w,2);
    if (w > 32*6*64) then
    begin
      TsvSoundObject.done;
      fail;
    end;

    fillchar(data^,32*6*64,0);
    f.svFileIO(data^,w);
    line := 0;
    pos  := 0;

    while (line < 64) do
    begin
      if (pos < 32*6*64)and(data^[pos] <> 0) then
      begin
        b  := data^[pos];

        if (b and $20 <> 0) then
        begin
          if (data^[pos+2] <> 0) then
          channeluse[b and 31] := true;
          inc(pos,2);
        end;

        if (b and $40 <> 0) then inc(pos);
        if (b and $80 <> 0) then inc(pos,2);

      end else inc(line);
      inc(pos);
    end;
  end;

  playchnls := 0;
  fillchar(mapping,sizeof(mapping),$FF);
  for i := 0 to 31 do
  if (channeluse[i])and(header.channelinfo[i] and $80 = 0) then
  begin
    mapping[playchnls] := i;
    inc(playchnls);
  end;

  linesize    := playchnls*sizeof(tpatt);

  setchannels(playchnls);
  setbuffersize(round(dat.mainfreq/(startbpm*2/5)));

  fillchar(channeluse,sizeof(channeluse),0);
  { Decompress and convert pattern data }
  for i := 0 to patterns-1 do
  begin
    with pattern[i] do
    begin
      size := linesize*64;
      rows := 63;
      getmem(p,size);
    end;
    fillchar(pattern[i].p^,pattern[i].size,0);
    if (s3mpattpointer[i] <> 0) then
    begin
      f.Position:=(longint(s3mpattpointer[i])*16{,soFromBeginning});
      f.svFileIO(w,2);
      if (w > 32*6*64) then
      begin
        TsvSoundObject.done;
        fail;
      end;
      fillchar(data^,w,0);
      f.svFileIO(data^,w);
      line := 0;
      pos  := 0;

      while (line < 64) do
      begin

        if (pos < 32*6*64)and(data^[pos] <> 0) then
        begin
          b  := data^[pos];
          ch := mapping[b and 31];
          fillchar(patt,sizeof(patt),0);

          if (b and $20 <> 0) then
          begin
            case data^[pos+1] of
              255 : patt.per := 0;
              254 : patt.per := mdkeyoff;
              else
              begin
                x := data^[pos+1] shr 4; if (x > 10) then x := 10;
                y := data^[pos+1] and $F;if (y > 11) then y := 11;
                patt.per := s3mtab[x,y];
              end;
            end;
            patt.ins := data^[pos+2];
            inc(pos,2);
          end;

          if (b and $40 <> 0) then
          begin
            if (data^[pos+1] <= 64) then patt.vol := data^[pos+1]+$10;
            inc(pos);
          end;

          if (b and $80 <> 0) then
          begin
            if (ch <> $FF) then { Only insert if correct assigned }
            begin
              patt.arg := data^[pos+2];
              case data^[pos+1] of
                $01 : patt.cmd := $F5; { S3M set tempo }
                $02 : patt.cmd := $0B;
                $03 : patt.cmd := $0D;
                $04 : patt.cmd := $22; { S3M Volume slide }
                $05 : patt.cmd := $23; { S3M Porta dn       }
                $06 : patt.cmd := $24; { S3M Porta up       }
                $07 : patt.cmd := $03; { Porta to note      }
                $08 : patt.cmd := $04; { Vibrato            }
                $09 : patt.cmd := $1D; { Tremor             }
                $0A : patt.cmd := $00; { Arpeggio           }
                $0B : patt.cmd := $06; { Vibr+vol slide     }
                $0C : patt.cmd := $05; { Porta+vol slide    }
                $0F : patt.cmd := $09; { Set sample ofs     }

                $11 : patt.cmd := $1B; { Retrig+vol slide   }
                $12 : patt.cmd := $07; { Tremolo            }
                $13 : patt.cmd := $0E;

                $14 : patt.cmd := $0F; { Set tempo/bpm      }
                $15 : patt.cmd := $F4; { Extra fine vibrato }
                $16 : patt.cmd := $10; { Set global volume  }
                else
                begin
                  patt.cmd := 0;
                  patt.arg := 0;
                end;
              end;
              if (patt.cmd = $0E) then
              begin
                case (patt.arg shr 4) of
                  $0 : if (patt.arg <> 1)and(patt.arg <> 0) then
                       begin;patt.arg := 0;patt.cmd := 0;end;
                  $1 : patt.arg := (patt.arg and $F) + $30; { Set glissando        }
                  $2 : patt.arg := (patt.arg and $F) + $50; { Set finetune         }
                  $3 : patt.arg := (patt.arg and $F) + $40; { Set vibrato waveform }
                  $4 : patt.arg := (patt.arg and $F) + $70; { Set tremolo waveform }

                  { .... not defined }

                  { $9 not defined }
                  $A : begin { Stereo control - old panning }
                         if (patt.arg < 8) then inc(patt.arg,8) else
                         dec(patt.arg,8);

                         patt.arg := (patt.arg and $F) or $80;
                         {patt.cmd := $0E;}
                      end;
                  $B : begin { Pattern loop }
                         {patt.cmd := $0E;}
                         patt.arg := patt.arg and $F or $60;
                       end;

                  { 8,C..E don't require conversion }
                  { F ("Funk repeat") not supported by ST3 anyway... }
                end;
              end;
              if (patt.cmd = $1E) then patt.cmd := $0E;
            end;
            inc(pos,2);
          end;
          if (ch < playchnls) then
          move(patt,pointer(longword(pattern[i].p)+longword(
          line*linesize+ch*sizeof(patt)))^,sizeof(patt));
        end else inc(line);
        inc(pos);
      end;
    end;
  end;
  freemem(data,32*6*64+6);

  cs.init;
  cs.globalvol := header.globalvol*4;

  if (header.mastervol and $80 = 0)and(header.defpan <> $FC) then { Mono }
  for i := 0 to 31 do cs.channel[i].panning := 0
    else
  for i := 0 to 31 do if (mapping[i] < playchnls) then with cs.channel[mapping[i]] do
  begin
    if (panlist[i] and $20 = 0) then
    begin
      panning := (i and 1)*512-256;
      if (panning < 0) then
        inc(panning,51) else
        dec(panning,51);
    end
    else panning := integer(panlist[i] and $F)*32-256;
  end;

  for i := 0 to playchnls-1 do cs.channel[i].sound.init(self);


  { Load instrument samples }
  for i := 1 to instruments do
  begin
    f.Position:=(longint(s3minstpointer[i-1])*16{,soFromBeginning});
    f.svFileIO(s3minstrument,sizeof(ts3minstr));
    with cs.instrument[i] do
    begin
      f.Position:=(longint(s3minstrument.segment)*16{,soFromBeginning});
      size   := s3minstrument.size;
      name   := s3minstrument.samplename;
      ofreq  := stdfreq;
      tuning := s3minstrument.c2spd;
      if (s3minstrument.flags and 1 <> 0) then
      begin
        loopstart := s3minstrument.loopstart;
        loopend   := s3minstrument.loopend;
      end else
      begin
        loopstart := 0;
        loopend   := 0;
      end;
      volume  := s3minstrument.volume;
      volfade := 65536;

      if (cs.instrument[i].size > 0) then
      with cs.instrument[i] do
      begin
        if (s3minstrument.flags and $4 = 0) then { 8 bit data }
        begin
          getmem(s,size shl 1);
          getmem(p,size);
          f.svFileIO(p^,size);
          open := true;
          aud_us8_to_s16(p,s,size);
          freemem(p);
        end else
        begin
          getmem(s,size);
          f.svFileIO(s^,size);
          aud_us16_to_s16(s,s,size shr 1);
          open := true;
          size      := size shr 1;
          loopstart := loopstart shr 1;
          loopend   := loopend shr 1;
        end;
      end;
    end;
  end;
  initialized := true;
end;

constructor txm.init;
type
  ptinfo = packed record
    headersize : longint;
    pack       : byte;
    rows       : word;
    size       : word;
  end;

  tln = array[0..31] of tpatt;

  txmpatt = packed record
    note : byte;
    ins  : byte;
    vol  : byte;
    cmd  : byte;
    arg  : byte;
  end;

  txminshdr = packed record
    size       : longint;
    samplename : array[0..21] of char;
    itype      : byte;
    samples    : word;
  end;

  txmins2hdr = packed record
    headersize : longint;
    smpmap     : array[0..95] of byte;
    venvelope  : array[0..11] of record
                   x,y : Word;
                 end;
    penvelope  : array[0..11] of record
                   x,y : Word;
                 end;
    vpoints    : byte;
    ppoints    : byte;
    vsustain   : byte;
    vlstart    : byte;
    vlend      : byte;
    psustain   : byte;
    plstart    : byte;
    plend      : byte;
    vtype      : byte;
    ptype      : byte;
    vitype     : byte;
    visweep    : byte;
    videpth    : byte;
    virate     : byte;
    vfadeout   : word;
    reserved   : array[0..10] of byte;
  end;

  txmsamplehdr = packed record
    size       : longint;
    loopstart  : longint;
    looplength : longint;
    volume     : byte;
    finetune   : shortint;
    flags      : byte;
    panning    : byte;
    relnote    : shortint;
    reserved   : byte;
    samplename : array[0..21] of char;
  end;

var
  i,i2,j  : integer;
  fpos    : longint;
  info    : ptinfo;
  datap   : pointer;
  b       : byte;
  d       : longint;
  patt    : tpatt;
  pos,ch  : word;
  line    : word;
  ihdr1   : txminshdr;
  ihdr2   : txmins2hdr;
  p       : pointer;
  np      : longint;
  count   : longint;
  sample  : array[0..15] of record
              smp    : txmsamplehdr;
            end;
  currsample : word;
  x,memsize  : longint;

function getbyte : byte;
type
  tba = array[0..0] of byte;
begin
  getbyte := byte(pointer(longword(datap)+pos)^);
  inc(pos);
end;

begin
  inherited init(dest);

  f.svFileIO(startinfo,sizeof(startinfo));
  if (makestring(startinfo.id,17) <> 'Extended Module: ') then
  begin
    TsvSoundObject.done;
    fail;
  end;
  fpos := f.position;
  f.svFileIO(header,sizeof(header));
  f.svFileIO(arrange,256);
  f.Position:=(header.headersize+fpos{,sofrombeginning});
  songname := filtername(startinfo.songname,19);
  songlength  := header.songlength;
  instruments := header.instruments;
  patterns    := header.patterns;
  playchnls   := header.channels;
  linesize    := playchnls*sizeof(tpatt);

  { Initialize format specific variables }
  amigalimits := header.flags and 1 = 0;
  format      := fmt_xm;
  oscillator  := xm_freq;
  pershift    := 0;
  portashift  := 0;
  minslide    := 1;
  maxslide    := 2000;
  startbpm    := header.startbpm;
  starttempo  := header.starttempo;
  if (startbpm = 0) then startbpm := 125;
  if (starttempo = 0) then starttempo := 1;
  arrangespeed[0].tempo := starttempo;
  arrangespeed[0].bpm   := startbpm;
  getmem(smpmapping,sizeof(txmsmpmapping));

  { Set number of playback channels and BPM }
  setchannels(playchnls);
  setbuffersize(round(dat.mainfreq/(startbpm*2/5)));

  tracker := makestring(startinfo.tracker,20);
  cs.init;

  for i := 0 to playchnls-1 do cs.channel[i].sound.init(self);
  getmem(datap,playchnls*5*256);


  { Create recommended empty pattern }
  with pattern[patterns] do
  begin
    size := linesize*64;
    rows := 63;
    getmem(dummypatt,size);
    fillchar(dummypatt^,size,0);
    p := dummypatt;
  end;

  { Point all non-existing arrangements to empty pattern }
  for i := 0 to 255 do if (arrange[i] > patterns) then
  begin
    arrange[i] := patterns;
  end;


  { Load all patterns }
  for i := 0 to patterns-1 do
  begin
    np := f.position;
    f.svFileIO(info,sizeof(info));
    inc(np,info.headersize);
    f.Position:=(np{,sofrombeginning});
    with pattern[i] do if (info.size > 0) then
    begin
      size := linesize*info.rows;
      rows := info.rows-1;
      getmem(p,size);
    end;
    { If pattern empty, create empty pattern }
    if (info.size = 0) then
    begin
      pattern[i] := pattern[patterns];
    end
    else
    begin { Else decompress pattern }
      count := f.position;
      f.svFileIO(datap^,info.size);
      count := f.position-count;
      pos := 0;
      for line := 0 to info.rows-1 do
      for ch := 0 to playchnls-1 do if (pos < info.size) then
      begin
        b := getbyte;
        if (b and $80 <> 0) then
        begin
          fillchar(patt,sizeof(patt),0);
          { Packed note       }
          if (b and $01 <> 0) then
          begin
            patt.per := getbyte;
            if (patt.per = 97) then patt.per := mdkeyoff
            else if (patt.per > 96) then patt.per := 96;
          end;
          { Packed instrument }
          if (b and $02 <> 0) then patt.ins := getbyte;
          { Packed volume     }
          if (b and $04 <> 0) then patt.vol := getbyte;
          { Packed command    }
          if (b and $08 <> 0) then patt.cmd := getbyte;
          { Packed argument   }
          if (b and $10 <> 0) then patt.arg := getbyte;
        end else { No packed data }
        begin
          patt.per := b and $7F;
          if (patt.per = 97) then patt.per := mdkeyoff
          else if (patt.per > 96) then patt.per := 96;
          patt.ins := getbyte;
          patt.vol := getbyte;
          patt.cmd := getbyte;
          patt.arg := getbyte;
        end;
        move(patt,pointer(longint(pattern[i].p)+line*linesize+ch*sizeof(patt))^,sizeof(patt));
      end;
      if (pos <> info.size) then
      begin
        TsvSoundObject.done;
        fail;
      end;
    end;
  end;
  freemem(datap,playchnls*5*256);
  { Load all instruments }
  currsample := 1;
  for i := 0 to header.instruments-1 do
  begin
    np := f.position;
    f.svFileIO(ihdr1,sizeof(ihdr1));
    if (f.position-np < sizeof(ihdr1)) then
    begin
      TsvSoundObject.done;
      fail;
    end;

    inc(np,ihdr1.size);

    if (ihdr1.samples > 0) then
    begin
      f.svFileIO(ihdr2,sizeof(ihdr2));
      f.Position:=(np{,sofrombeginning});
      move(ihdr2.smpmap,smpmapping^[i+1],96);

      if (ihdr1.samples > 16) then
      begin
        TsvSoundObject.done;
        fail;
      end;
      for j := 0 to ihdr1.samples-1 do
      begin
        count := f.position;
        f.svFileIO(sample[j].smp,sizeof(sample[j].smp));
        count := f.position-count;
      end;

      for j := 0 to ihdr1.samples-1 do with cs.instrument[currsample] do
      with sample[j] do
      begin
        { Restart if sample empty }
        if (smp.size = 0)and(j > 0) then
        begin
          continue;
        end;
        mapping[i+1,j] := currsample;
        inc(currsample);
        { Convert sample name }
        name := '';
        for i2 := 0 to 21 do
        if (ihdr1.samplename[i2] >= #32) then
          name := name+ihdr1.samplename[i2] else name := name+#32;
        if (smp.size = 0) then continue;

        { Set sample variables }
        volume  := smp.volume;
        relnote := smp.relnote;
        panning := (integer(smp.panning)-128)*2;
        ofreq   := 8363;
        tuning  := smp.finetune;
        size    := smp.size;
        volfade := ihdr2.vfadeout;

        { << Added by Raid >> }

        Envelope [0].Points    := IHDR2.VPoints;
        Envelope [0].Sustain   := IHDR2.VSustain;
        Envelope [0].LStart    := IHDR2.VLStart;
        Envelope [0].LEnd      := IHDR2.VLEnd;
        Envelope [0].EStat     := IHDR2.VType;

        Envelope [1].Points    := IHDR2.PPoints;
        Envelope [1].Sustain   := IHDR2.PSustain;
        Envelope [1].LStart    := IHDR2.PLStart;
        Envelope [1].LEnd      := IHDR2.PLEnd;
        Envelope [1].EStat     := IHDR2.PType;

        Move (IHDR2.VEnvelope, Envelope [0].EData, SizeOF (IHDR2.VEnvelope) );
        Move (IHDR2.PEnvelope, Envelope [1].EData, SizeOF (IHDR2.PEnvelope) );

        { << End >> }

        { Apply auto vibrato settings }
        avitype  := ihdr2.vitype;
        avisweep := ihdr2.visweep;
        avidepth := ihdr2.videpth;
        avispeed := ihdr2.virate;

        { Set looping
          0 : no looping
          1 : normal looping
          2 : ping-pong
        }

        memsize := size;
        if (smp.flags and 3 <> 0) then
        begin
          if (smp.loopstart+smp.looplength*2 > size) then
          memsize := smp.loopstart+smp.looplength*2;
        end;
        { Load and convert sample data }
        if (smp.size > 0) then
        begin

          if (smp.flags and $10 = 0) then { 8 bit data }
          begin
            getmem(s,memsize shl 1);
            getmem(p,size);
            count := f.position;
            f.svFileIO(p^,size);
            count := f.position-count;
            if (count <> size) then
            begin
              freemem(s);
              freemem(p);
              size := 0;
              open := false;
              {continue;}
              TsvSoundObject.done;
              fail;
            end
            else
            open := true;
            delta8(p,size);
            aud_s8_to_s16(p,s,size);
            freemem(p);
          end else
          begin  { 16 bit data }
            getmem(s,memsize);
            count := f.position;
            f.svFileIO(s^,size);
            count := f.position-count;
            if (count <> size) then
            begin
              freemem(s);
              size := 0;
              open := false;
              {continue;}
              TsvSoundObject.done;
              fail;
            end
            else
            open := true;
            isword := true; { req'd for set sample ofs }
            size := size shr 1;
            memsize := memsize shr 1;
            smp.loopstart  := smp.loopstart shr 1;
            smp.looplength := smp.looplength shr 1;
            delta16(s,size);
          end;
        end; { Load sample data }

        if (smp.flags and 3 <> 0)and(smp.looplength > 0) then
        begin
          loopstart := smp.loopstart;
          loopend   := smp.loopstart+smp.looplength;
          if (smp.flags and 3 = 2) then
          begin { Apply pingpong-loop }
            for x := 0 to smp.looplength-1 do
            word(pointer(longword(s)+longword((loopend+smp.looplength-x-1)*2))^) :=
                        word(pointer(longword(s)+longword((x+loopstart)*2))^);
            inc(loopend,smp.looplength);
            size := memsize;
          end;
        end else
        begin
          loopstart := 0;
          loopend   := 0;
        end;

      end; { Load samples }
    end else { Empty instrument }
    begin
      inc(currsample);
      f.Position:=(np{,sofrombeginning});
    end;
  end; { Load instruments }

  instruments := currsample-1;
  initialized := true;
end;

destructor txm.done;
begin
  inherited done;
  if assigned(smpmapping) then freemem(smpmapping,sizeof(txmsmpmapping));
end;

function txm.getfrequency(period : longword) : longword;
begin
  Result := oscillator div period
end;


procedure txm.calcline;
type
  tln = array[0..31] of tpatt;


var
  ln        : ^tln;
  w         : word;
  ch,i      : integer;
  dontplay  : boolean;
  curr      : tpatt;
  per       : longint;
  last      : word;
  patt_break,patt_jump : boolean;
  dontinc   : boolean; { Don't increase line at end of calcline }
  val       : real;

{ << Added by Raid >> }

Procedure SetupEnvelope(Ins: Integer);
Begin
  With CS.Channel [CH] DO
  Begin

{    IF CS.Instrument[Ins].Envelope[0].EStat and 1 <> 0 Then}
    Begin
      EnvelopeCH [0].Envelope := CS.Instrument[Ins].Envelope[0];
      EnvelopeCH [0].Active   := True;
      EnvelopeCH [0].CurY     := EnvelopeCH[0].Envelope.EData[0].Y;
      EnvelopeCH [0].CurPoint := 0;
      EnvelopeCH [0].ECounter := 0;
    End; { IF }


{    IF CS.Instrument[Ins].Envelope[1].EStat and 1 <> 0 Then}
    Begin
      EnvelopeCH [1].Envelope := CS.Instrument[Ins].Envelope[1];
      EnvelopeCH [1].Active   := True;
      EnvelopeCH [1].CurY     := EnvelopeCH[1].Envelope.EData[0].Y;
      EnvelopeCH [1].CurPoint := 0;
      EnvelopeCH [1].ECounter := 0;
    End; { IF }

  End; { With }
End; { Procedure }

{ << End >> }

begin

  if (actline = longword(pattern[arrange[actpattern]].rows)+1) then
  begin
    actline := 0;
    inc(actpattern);
    arrangespeed[actpattern].bpm   := bpm;
    arrangespeed[actpattern].tempo := tempo;
  end;

  if (actpattern >= songlength) then
  begin
    stop;
    exit;
  end;

  ln := pointer(longword(pattern[arrange[actpattern]].p)+longword(actline*linesize));

  patt_break := false; { Clear pattern break flag }
  patt_jump  := false; { Clear pattern jump flag  }
  dontinc    := false;


  for ch := 0 to playchnls-1 do with cs.channel[ch] do
  begin
    curr := ln^[ch];
    if (curr.ins <> 0)and(curr.per < 97)and(curr.per > 0) then
      curr.ins := mapping[curr.ins,smpmapping^[curr.ins,curr.per] and 15];
    { << Rem'd by Raid >> }
      {else curr.ins := 0;}
    { << End >> }
    per  := curr.per;

    { Ignore porta to note if instrument different }
    if (curr.cmd = $03)and(curr.ins <> 0)and(curr.ins <> voice) then
    begin
      curr.cmd := 0;
      curr.arg := 0;
    end;

    IF (Per <> 0) and (Per <> MDKeyOff) Then
    Begin
      w := curr.ins;
      if (w = 0) then w := voice;
      if (w = 0) then continue;
      { Original routine }
      {per := Round((xmtab[i]*(1-frac(cs.instrument[w].tuning div 16))+
                    xmtab[i+1]*(frac(cs.instrument[w+1].tuning div 16)))
                    *16/exp((per div 12)*system.ln(2)));}

      { Check if we should use Amiga table or linear calculation }
      if (amigalimits) then
      begin
        Inc (Per, CS.Instrument[W].RelNote);
        I := (Per mod 12) shl 3 + (CS.Instrument[W].Tuning div 16);
        IF (I < 0) Then I := 0 Else IF (I > 119) Then I := 119;
        Per := (XMTab[I] shr 1+XMTab[I+1] shr 1) shl 4 shr (Per div 12);
      end

      else { Do it linear... }

      begin
        val := per+CS.Instrument[W].RelNote+CS.Instrument[W].Tuning/128;
        if (val < 0) then val := 0
        else
        if (val > 119) then val := 119;
        per := round(907*exp(-(val)*system.ln(1.059463094))*16);
        {per := round(10*12*16*4 - (per+CS.Instrument[W].RelNote)*16*4 - cs.instrument[w].tuning/2);
        if (per < 1) then val := 1;}
      end;

      IF (Per <= 0) Then Per := 1;
      if (curr.cmd <> $03) then
      begin
        period := per;
        freq   := {oscillator div per;}getfrequency(per);
      end;
    End; { IF }

    { Don't play note if note delay, porta to note or key off }
    dontplay := ((curr.cmd = $0E)and((curr.arg shr 4) = $0D))or
                 ((curr.cmd = $03){and((curr.ins = 0)or(curr.ins = voice))})or(curr.per = mdkeyoff);

    { Strange thing... reset volume if instrument, but not period given }
    if (curr.ins <> 0)and(curr.per = 0)and(voice <> 0) then
    begin
      volume := cs.instrument[voice].volume;
      cs.setvolume(ch,volume);

      {!! Not sure if this should reset panning and envelope as well }

    {  Panning := CS.Instrument[Voice].Panning;
      SetupEnvelope(Voice);}
    end

    else

    if (not dontplay) then
    if (curr.ins <> 0) then
    begin
      Panning := CS.Instrument[Curr.Ins].Panning;
      SetupEnvelope(Curr.Ins);
      CS.Play(CH, Curr.Ins, Freq, CS.Instrument[Curr.Ins].Volume)
    end
    else if (per <> 0)or(curr.ins <> 0)and(per = 0) then
    begin
      Panning := CS.Instrument[Voice].Panning;
      SetupEnvelope(Voice);
      CS.Play(CH, Voice, Freq, Volume);
    end;


    { Retrigger volume if porta note and same instrument on line }
    if (curr.cmd = $03)and(curr.ins <> 0)and(curr.ins = voice)and
       (format <> fmt_mod)and(curr.vol = 0)
       then curr.vol := cs.instrument[voice].volume+$10;

    if ((arpeggio)or(vibrato))and(period <> 0) then
    begin
      cs.setfrequency(ch,freq);
      arpeggio    := false;
      vibrato     := false;
    end;

    { Reset all FX which are linewise }
    if (amigalimits) then porta := false;
    tremolo      := false;
    bfinepslide  := false;
    bfinevslide  := false;
    bvolumeslide := false;
    bportaslide  := false;
    BPanSlide    := false;
    retrig       := 0;
    tremor       := false;
    last         := lasteffect;
    lasteffect   := curr.cmd+curr.arg shl 8;

    if (curr.vol <> 0) then
    begin
      if (curr.vol <= $50) then
      begin
        if (curr.vol >= $10) then volume := curr.vol-$10;
        if ((curr.cmd = $0E)and((curr.arg shr 4) = $0D))
        then else cs.setvolume(ch,volume);
      end else
      begin
         { XM FX... not complete, haven't had good docu }
         case (curr.vol shr 4) of
           $6 : begin;volumeslide := -(curr.vol and $F);bvolumeslide := true;end;
           $7 : begin;volumeslide := (curr.vol and $F);bvolumeslide := true;end;
           { << Fine Volume Sliding Effects in Volume Column >> }
           $8 : Begin; { // Fine Volume Slide Down // }
                  FineVSlide  := -(Curr.Vol And $F);
                  BFineVSlide := True;
                End;
           $9 : Begin; { // Fine Volume Slide Up // }
                  FineVSlide  := (Curr.Vol And $F);
                  BFineVSlide := True;
                End;
           { << End >> }
           { $A - set vibrato speed }
           { $B - vibrato }
           $C : begin
                  panning := (curr.vol and $F);
                  if (panning > 7) then inc(panning);
                  panning := panning shl 5 - 256;
                  cs.setvolume(ch,cs.channel[ch].volume);
                end;
           $D : begin
                  if (panning-curr.vol and $F > -256) then
                    dec(panning,curr.vol and $F) else panning := -256;
                  cs.setvolume(ch,cs.channel[ch].volume);
                end;
           $E : begin
                  if (panning+curr.vol and $F < 256) then
                    inc(panning,curr.vol and $F) else panning := 256;
                  cs.setvolume(ch,cs.channel[ch].volume);
                end;
           { $F - tone porta }
         end;
      end;
    end;

    { Check for key off }
    IF (Curr.Per = MDKeyOff) Then
    Begin
      { // Key Off // }
      if (voice <> 0) then
      IF CS.Instrument [Voice].Envelope [0].EStat and 1 = 0 Then
      Begin
        Volume := 0; CS.SetVolume (CH, Volume);
      End { IF }
      Else KeyFading := True;
    End; { IF }

    { Process effects }
    case curr.cmd of
      $00 : if (curr.arg <> 0) then
            begin { Arpeggio          }
              arpeggio  := true;
              arpeggiox := (curr.arg shr 4);
              arpeggioy := (curr.arg and $F);
            end else lasteffect := mdkeyoff;
      $01 : begin { Portamento up     }
              if (curr.arg <> 0) then
                porta_up  := -(integer(curr.arg) shl portashift);
              if (porta_up <> 0) then
              begin
                bportaslide := true;
                portaslide  := porta_up;
              end;
            end;
      $02 : begin { Portamento down   }
              if (curr.arg <> 0) then
                porta_dn  := integer(curr.arg) shl portashift;
              if (porta_dn <> 0) then
              begin
                bportaslide := true;
                portaslide  := porta_dn;
              end;
            end;
      $03 : begin { Porta to note     }
              IF (Curr.Ins <> 0) Then SetupEnvelope(Curr.Ins);
              if (per <> 0) then portanote := per;
              if (curr.arg <> 0) then
              begin
                portaspeed := word(curr.arg) shl portashift;
                if (period > portanote) then portaspeed := -portaspeed;
              end;
              porta := (portanote <> 0);
            end;
      $04 : begin { Set vibrato       }
              if (curr.arg and $F0 <> 0) then vibratospeed := curr.arg shr 4;
              if (curr.arg and $F <> 0) then vibratodepth := (curr.arg and $F) shl pershift;
              vibrato := true;
            end;
      $05 : begin { Porta+vol slide   }
              if (curr.arg and $F <> 0) then volumeslide := -(curr.arg and $F)
                else if (curr.arg and $F0 <> 0) then volumeslide := curr.arg shr 4;
              porta := (portanote <> 0);
              bvolumeslide := true;
            end;
      $06 : begin { Vibrato+vol slide }
              if (curr.arg and $F <> 0) then volumeslide := -(curr.arg and $F)
              else if (curr.arg and $F0 <> 0) then volumeslide := curr.arg shr 4;
              vibrato := true;
              bvolumeslide  := true;
            end;
      $07 : begin { Tremelo           }
              if (curr.arg and $F0 <> 0) then tremolospeed := curr.arg shr 4;
              if (curr.arg and $F <> 0) then tremolodepth := curr.arg and $F;
              tremolo := true;
            end;
      $08 : begin { Panning           }
              panning := integer(curr.arg) shl 1 - 255;
              CS.SetVolume(CH,CS.Channel[CH].Volume);
            end;
      $09 : begin { Set instrument pos }
              if (curr.ins <> 0) Then
              begin
                if (curr.arg <> 0) Then
                if cs.instrument[curr.ins].isword then
                lastsetpos := longint(curr.arg) shl 7
                else
                lastsetpos := longint(curr.arg) shl 8;
                cs.setpos(ch,lastsetpos);
              end;
            end;
      $0A : begin
              if (curr.arg and $F <> 0) then volumeslide := -(curr.arg and $F)
              else if (curr.arg and $F0 <> 0) then volumeslide := curr.arg shr 4;
              bvolumeslide  := (volumeslide <> 0);
            end;
      $0B : { Pattern jump }
            begin
              if (curr.arg > songlength-1) then curr.arg := 0; { Means 'restart' - could stop here }
              actline    := 0;
              actpattern := curr.arg;
              dontinc    := true;
              patt_jump  := true;
            end;
      $0C : begin { Set channel volume }
              if (curr.arg > 64) then curr.arg := 64;
              cs.channel[ch].volume := curr.arg;
              cs.setvolume(ch,curr.arg);
            end;
      $0D : begin { Pattern break      }
              if (not patt_jump)and(not patt_break) then
                if (actpattern < songlength-1) then inc(actpattern) else actpattern := 0;
              actline := 10*(curr.arg shr 4)+curr.arg and $0F;
              if (actline > 63) then actline := 0;
              dontinc    := true;
              patt_break := true;
            end;
      $0E : case (curr.arg shr 4) of
              $01 : begin { ok; Fine portamento up   }
                      if (curr.arg and $F <> 0) then
                        finepslide := -((curr.arg and $F) shl portashift);
                      bfinepslide := (finepslide <> 0);
                    end;
              $02 : begin { ok; Fine portamento down }
                      if (curr.arg and $F <> 0) then
                        finepslide := (curr.arg and $F) shl portashift;
                      bfinepslide := (finepslide <> 0);
                    end;
              $03 : glissando := curr.arg and 1 = 1; { glissando control }
              $04 : begin { Set vibrato waveform }
                      vibratotype := curr.arg and 7;
                      if (vibratotype and 3) = 3 then
                        vibratotype := random(2)+curr.arg and 4;
                      {!! Note: random seems not to be supported in XM }
                    end;
              $05 : begin { Set finetune }
                      { This is wrong! Fine tuning should affect the XM style tuning... }
                      {cs.instrument[voice].tuning := getc2spd(curr.arg and $F);}
                      cs.instrument[voice].tuning := shortint(curr.arg);
                    end;
              $06 : if (not seeking) then
                    if (not patternloop.active) then  { pattern loop }
                    begin
                       if (curr.arg and $F = 0) then { We have to set the loop start }
                       begin
                         patternloop.line     := actline;
                         patternloop.pattern  := actpattern;
                       end else { we have the loop end }
                       begin
                         patternloop.active   := true;     { This causes that a new set of loop start overwrites old one }
                         patternloop.count    := curr.arg and $F;
                         actline              := patternloop.line;
                         dontinc              := true;
                       end;

                    end else
                    if (curr.arg and $F <> 0) then { We already are repeating, do jump/count down }
                    begin
                       dec(patternloop.count);
                       if (patternloop.count > 0) then
                       begin
                         actline := patternloop.line;
                         dontinc := true;
                       end else patternloop.active := false;
                    end;
              $07 : begin { Set tremolo waveform }
                      tremolotype := curr.arg and 7;
                      if (tremolotype and 3) = 3 then
                        tremolotype := random(2)+curr.arg and 4;
                    end;
              $08 : begin { ok; Position panning }
                      panning := (curr.arg and $F);
                      if (panning > 7) then inc(panning);
                      panning := panning shl 5 - 256;
                      cs.setvolume(ch,cs.channel[ch].volume);
                    end;
              $09 : begin { ok; Retrigger note   }
                      retrig     := (curr.arg and $F);
                      retrigtype := 0;
                    end;
              $0A : begin { ok; Fine volume slide up   }
                      if (curr.arg and $F <> 0) then
                        finevslide := curr.arg and $F;
                      bfinevslide := (finevslide <> 0);
                    end;
              $0B : begin { ok; Fine volume slide down }
                      if (curr.arg and $F <> 0) then
                        finevslide := -integer(curr.arg and $F);
                      bfinevslide := (finevslide <> 0);
                    end;
              $0C : begin { ok; Cut note }
                      if (curr.arg and $F <> 0) then volcutcount := (curr.arg and $F);
                    end;
              $0D : begin { ok; Note delay }
                      notedelay := (curr.arg and $F);
                      if (curr.ins <> 0) then voice := curr.ins;
                      if (curr.vol = 0) then volume := cs.instrument[voice].volume;
                    end;
              $0E : begin { ok; Pattern delay }
                      patdelay := (curr.arg and $F);
                    end;
              $0F :; { invert loop; not supported anyway }
            end;
      $0F : if (curr.arg <> 0) then
            begin{ok}
              if (curr.arg < 32) then tempo := curr.arg
              else
              begin
                bpm := curr.arg;
                setbuffersize(round(dat.mainfreq/(bpm shl 1 / 5)));
              end;
            end;
      $10 : begin { Set global volume }
              if (curr.arg > 64) then curr.arg := 64;
              cs.globalvol := curr.arg shl 2;
              for i := 0 to playchnls-1 do cs.setvolume(i,cs.channel[i].volume);
            end;
      $11 : Begin { Global volume slide }
              if (curr.arg <> 0) then CS.GVolSlide := (Curr.Arg Shr 4) - (Curr.Arg And $F);

              CS.GVolCounter := CS.GVolSlide;
              IF CS.GVolCounter < 0 Then
                 CS.GVolCounter := Not CS.GVolCounter + 1;
              CS.BGVolSlide  := (CS.GVolSlide <> 0);
            End; { H }
      $14 : begin { Key off }
              volume := 0;
              cs.setvolume(ch,volume);
            end;


      $15 : ; { set envelope pos  }

      { !! shouldn't this be $18 ?? Triton says 19, XM doc 0.81 says 18 }
      { << Panning Slide >> }
      $18, $19 : Begin
             if (curr.arg <> 0) then PanSlide   := (Curr.Arg Shr 4) - (Curr.Arg And $F);
             PanCounter := PanSlide;
             IF PanCounter < 0 Then PanCounter := Not PanCounter + 1;
             BPanSlide  := (PanSlide <> 0);
           End; { P }
      { << End >> }

      {!! same goes here...  }
      $1A, $1B : begin
              if (curr.arg <> 0) then lretrig := curr.arg;
              retrig     := (lretrig and $F);
              retrigtype := (lretrig shr 4);
            end;
      $1D : begin { Tremor }
              tremor      := true;
              tremorcount := 0;
              tremorx     := curr.arg shr 4+1;
              tremory     := tremorx+curr.arg and 7+1;
            end;

      $21 : case (curr.arg shr 4) of
              $01 :  begin { ok; Extra fine portamento up   }
                      if (curr.arg and $F <> 0) then
                        finepslide := -(curr.arg and $F);
                      bfinepslide := (finepslide <> 0);
                    end;
              $02 : begin { ok; Extra Fine portamento down }
                      if (curr.arg and $F <> 0) then
                        finepslide := curr.arg and $F;
                      bfinepslide := (finepslide <> 0);
                    end;
            end;
      else lasteffect := mdnotavail;
    end;
  end;

  if (not dontinc) then
  begin
    inc(actline);
    { Reset pattern loop data if required }
    if (actline = 0)or(seeking) then for ch := 0 to playchnls-1 do with cs.channel[ch] do
      patternloop.active := false;
  end;
end;

end.
